#ifndef _BUFFER_HPP
#define _BUFFER_HPP 1
#include <vector>
#include <mutex>
#include <queue>
#include <string>
#include "JobTool.hpp"
#include "Semaphore.hpp"
#include "thrift/gen-cpp/Remote_types.h"
using namespace std;

struct BufferCmp {
	// to get an max-heap we have to implement lhs >= rhs
	bool operator()(const Job & lhs, const Job & rhs) {
		// maximierung: jobs mit hoeherer Simplex-Loesung werden bevorzugt
		if (JobTool::BAD_OptNBound < 0.0) {
			return (lhs._bestBound >= rhs._bestBound);
		} else {
			//sonst: minimierung
			return (lhs._bestBound <= rhs._bestBound);
		}
	}
};

class Buffer {
private:
	Obj * _in;
	Configurator * _conf;
	
	vector<priority_queue<Job, vector<Job>, BufferCmp> > _values;
	int _deepness;
	int _limit;
	
	Semaphore * _filled;
	vector<long> _enlargeCount;
	mutex _mu;

public:
	// =================================================================
	Buffer(Obj * in, Configurator * conf) {
		_filled = new Semaphore(0);
		_in = in;
		_conf = conf;

		unsigned n = _in->cols.size();
		_enlargeCount.resize(n, 0);
		_deepness = n+1;
		_values.resize(_deepness);
		_limit = (_conf->breadthEndProcent/100.0) * n;
	}

	// =================================================================
	virtual ~Buffer() {
		delete _filled;
	}

	// =================================================================
	void init() {
		Job job = JobTool::create(_conf->jobTimeout);
		// nur ein job zu anfang
		_mu.lock();
		_enlargeCount[0] = 0;
		_values[0].push(job);
		_mu.unlock();
		_filled->V();
	}

	// =================================================================
	Job pop() {
		_filled->P();
		_mu.lock();
		Job _return;

		// -> erst jobs mit höchster Vorbelegung machen bis _limit
		bool found = false;
		for (int i = _deepness; i > _limit; --i) {
			if (_values[i-1].empty() == false) {
				_return = _values[i-1].top();
				_values[i-1].pop();
				found = true;
				break;
			}
		}
		// Gibt es keine jobs mit _limit als Vorbelegung mehr?
		// -> Dann Jobs mit wenig Vorbelegung zuerst nehmen !
		if (found == false) {
			for (int i = 0; i < _limit; i++) {
				if (_values[i].empty() == false) {
					_return = _values[i].top();
					_values[i].pop();
					break;
				}
			}
		}
		_mu.unlock();

		JobTool::clearBestBound(_return);
		return _return;
	}

	// =================================================================
	void finish(const int & allWorker) {
		Job job = JobTool::createEnd();
		for (int t = 0; t < allWorker; ++t) {
			_mu.lock();
			_values[0].push(job);
			_mu.unlock();
			_filled->V();
		}
	}

	// =================================================================
	void push(const vector<Job> & jobs) {
		int u = JobTool::deepness(jobs.at(0));
		_mu.lock();
		_enlargeCount[jobs.at(0)._lastK] += 1;
		_mu.unlock();

		for (auto j : jobs) {
			_mu.lock();
			_values[u].push(j);
			_mu.unlock();
			_filled->V();
		}
	}

	// =================================================================
	vector<long> size() {
		_mu.lock();
		vector<long> ret;
		for (auto & v : _values)
			ret.push_back(v.size());

		_mu.unlock();
		return ret;
	}

	// =================================================================
	vector<long> getEnlargeCount() {
		_mu.lock();
		vector<long> ret = _enlargeCount;
		_mu.unlock();
		return ret;
	}
};

#endif
