#include "thrift/gen-cpp/Remote_types.h"
#include "Worker_impl.hpp"
#include <iostream>
#include <ctime>
#include <chrono>
#include <mutex>
#include <glpk.h>
#include <vector>
#include <string>
#include <limits>
#include <unistd.h>
#include "JobTool.hpp"

using namespace std::chrono;

bool Worker_impl::_kill;
Configurator Worker_impl::_conf;
std::mutex Worker_impl::_mu;
Job Worker_impl::_bestJob;
double Worker_impl::_bestBound;
bool Worker_impl::_silent;
glp_prob * Worker_impl::_lp;
Obj Worker_impl::_data;

int Worker_impl::_megaMemLimit; // fuer logging

int Worker_impl::_jobLimit;
steady_clock::time_point Worker_impl::_startPoint;
int Worker_impl::TIMEOUT_REACHED = -42;


// callback function to the mip!! ======================================
void influence(glp_tree * tr, void * info) {

	// kill the mip/job/thread -----------------------------------------
	if(Worker_impl::_kill == true) {
		glp_ios_terminate(tr);
		return;
	}

	// timeout ---------------------------------------------------------
	if(Worker_impl::_jobLimit > 0) {
		steady_clock::time_point t1;
		t1 = steady_clock::now();
		if(duration_cast<milliseconds>(t1 - Worker_impl::_startPoint).count() > Worker_impl::_jobLimit) {
			Worker_impl::_jobLimit = Worker_impl::TIMEOUT_REACHED;
			
			glp_ios_terminate(tr);
			return;
		}
	}

	// -----------------------------------------------------------------
	Job best = *((Job *) info);
	double prof;

	switch (glp_ios_reason(tr)) {
		// is that reason a good choice ?
		case GLP_IHEUR:
			prof = glp_mip_obj_val(Worker_impl::_lp);
			Worker_impl::_mu.lock();
			if (
				(JobTool::compare(Worker_impl::_bestJob, prof) < 0 &&  Worker_impl::_data.isMax) ||
				(JobTool::compare(Worker_impl::_bestJob, prof) > 0 && !Worker_impl::_data.isMax)
			) {

				//store new bestState ----------------------------------
				best._fix.resize(Worker_impl::_data.cols.size());
				for(unsigned i=0; i<Worker_impl::_data.cols.size(); ++i) {
					best._fix[i] = glp_mip_col_val(Worker_impl::_lp, i+1);
				}
				best._opt = prof;
				best._end = false;
				best._bestBound = JobTool::BAD_OptNBound;
				best._timeout = Worker_impl::_jobLimit;
				Worker_impl::_bestJob = best;
			} else {
				// Update: bestState constraint ------------------------

// Es existiert eine (nicht initale) Schranke fuer aktuellen Job UND
// globales Maximum ist groesser als obere Schrank des akt. Jobs (bei Maximierung) BZW
// globales Minimum ist kleiner als untere Schrank des akt. Jobs (bei Minimierung)
				if (
					Worker_impl::_bestBound != JobTool::BAD_OptNBound && (
						(JobTool::compare(Worker_impl::_bestJob, Worker_impl::_bestBound) >= 0 &&  Worker_impl::_data.isMax) ||
						(JobTool::compare(Worker_impl::_bestJob, Worker_impl::_bestBound) <= 0 && !Worker_impl::_data.isMax)
					)
				) {
// -> wir koennen aktuellen Job abbrechen
					Worker_impl::_mu.unlock();
					glp_ios_terminate(tr);
					return;
				}
				
				int T = Worker_impl::_data.rows.size();
				
				if (Worker_impl::_data.isMax == true) {
					glp_set_row_bnds(
						Worker_impl::_lp,
						T+1, 
						GLP_DB, 
						Worker_impl::_bestJob._opt, 
						Worker_impl::_bestBound
					);
				} else {
					glp_set_row_bnds(
						Worker_impl::_lp,
						T+1, 
						GLP_DB, 
						Worker_impl::_bestBound,
						Worker_impl::_bestJob._opt
					);
				}
			}
			Worker_impl::_mu.unlock();
			break;

		default:
			break;
	}
}


// =====================================================================
void Worker_impl::solve(SolveResult & _sreturn, const Job & bestJob, const Job & newJob) {
	
	_sreturn.newJobs.push_back(newJob);
	Worker_impl::_jobLimit = newJob._timeout;
	
	Worker_impl::_mu.lock();
	if (
		(JobTool::compare(Worker_impl::_bestJob, bestJob) < 0 &&  Worker_impl::_data.isMax) ||
		(JobTool::compare(Worker_impl::_bestJob, bestJob) > 0 && !Worker_impl::_data.isMax)
	) {
		Worker_impl::_bestJob = bestJob;
	}
	Worker_impl::_mu.unlock();

	if (Worker_impl::_kill == true) {
		_sreturn.flag = SolveFlag::KILLED;
		return;
	}

	// SOLVE START -----------------------------------------------------
	int n  = Worker_impl::_data.cols.size();
	int T  = Worker_impl::_data.rows.size();

	if (Worker_impl::_jobLimit > 0) {
		Worker_impl::_startPoint = steady_clock::now();
	}
	
	// Bounds ----------------------------------------------------------
	Worker_impl::_mu.lock();
	
	if (Worker_impl::_data.isMax == true) {
		glp_set_row_bnds(Worker_impl::_lp, T+1, GLP_LO, Worker_impl::_bestJob._opt, 0.0);
	} else {
		glp_set_row_bnds(Worker_impl::_lp, T+1, GLP_UP, 0.0, Worker_impl::_bestJob._opt);
	}
	if (
		_sreturn.newJobs[0]._bestBound != JobTool::BAD_OptNBound && (
			(JobTool::compare(Worker_impl::_bestJob, _sreturn.newJobs[0]._bestBound) >= 0 &&  Worker_impl::_data.isMax) ||
			(JobTool::compare(Worker_impl::_bestJob, _sreturn.newJobs[0]._bestBound) <= 0 && !Worker_impl::_data.isMax)
		)
	) {
		// the upper (best)Bound should not be lower than the lowerBound (Best Solution/Job)
		Worker_impl::_mu.unlock();
		_sreturn.flag = SolveFlag::BOUNDERR;
		return;
	}
	Worker_impl::_mu.unlock();
	
	// lazy constraints -----------------------
	
	// auf standard setzen
	for (auto col : Worker_impl::_data.cols) {
		glp_set_col_kind(Worker_impl::_lp, col.at, col.kind);
		glp_set_col_bnds(Worker_impl::_lp, col.at, col.type, col.lower, col.upper);
	}
	// spezielle anpassen
	for (auto col : _sreturn.newJobs[0]._lazy) {
		glp_set_col_kind(Worker_impl::_lp, col.at, col.kind);
		glp_set_col_bnds(Worker_impl::_lp, col.at, col.type, col.lower, col.upper);
	}

	// simplex ---------------------------------------------------------
	std::vector<double> sim;
	
	if (Worker_impl::_jobLimit > 0) { // timelimit? -> no presolver

		glp_simplex(Worker_impl::_lp, &_parm1);
		
		switch (glp_get_status(Worker_impl::_lp)) {
			case GLP_NOFEAS:
				_sreturn.flag = SolveFlag::NOSOL;
				return;

			case GLP_UNDEF:
				/// @todo das muss den Farmer eigentlich abbrechen lassen!!?
				_sreturn.flag = SolveFlag::FAIL;
				return;
		}
		
		// simplex als bisher (niedrigste obere) Grenze = upper (best)Bound
		_sreturn.newJobs[0]._bestBound = glp_get_obj_val(Worker_impl::_lp); 

		// fix bekommt erstmal simplex loesung
		for (int i = 1; i <= n; ++i) {
			sim.push_back(glp_get_col_prim(Worker_impl::_lp, i));
		}
	}
	
	// Bound correction ------------------------------------------------
	Worker_impl::_bestBound = _sreturn.newJobs[0]._bestBound;
	Worker_impl::_mu.lock();
	
	if (Worker_impl::_bestBound != JobTool::BAD_OptNBound) {
		if (Worker_impl::_data.isMax == true) {
			if (JobTool::compare(Worker_impl::_bestJob, _sreturn.newJobs[0]._bestBound) >= 0) {
				// the upper (best)Bound should not be lower than the lowerBound (Best Solution/Job)
				Worker_impl::_mu.unlock();
				_sreturn.flag = SolveFlag::BOUNDERR;
				return;
			}
			// setzen beider Schranken
			glp_set_row_bnds(Worker_impl::_lp, T+1, GLP_DB, Worker_impl::_bestJob._opt, Worker_impl::_bestBound);

		} else {
			if (JobTool::compare(Worker_impl::_bestJob, _sreturn.newJobs[0]._bestBound) <= 0) {
				// the upper (best)Bound should not be lower than the lowerBound (Best Solution/Job)
				Worker_impl::_mu.unlock();
				_sreturn.flag = SolveFlag::BOUNDERR;
				return;
			}
			// setzen beider Schranken
			glp_set_row_bnds(Worker_impl::_lp, T+1, GLP_DB, Worker_impl::_bestBound, Worker_impl::_bestJob._opt);
		}
	} else {
		// obere Bzw. untere Schranke setzten
		if (Worker_impl::_data.isMax == true) {
			glp_set_row_bnds(Worker_impl::_lp, T+1, GLP_LO, Worker_impl::_bestJob._opt, 0.0);
		} else {
			glp_set_row_bnds(Worker_impl::_lp, T+1, GLP_UP, 0.0, Worker_impl::_bestJob._opt);
		}
	}
	Worker_impl::_mu.unlock();

	// MIP -------------------------------------------------------------
	glp_init_iocp(&_parm2);
	_parm2.cb_func = influence;
	_parm2.cb_info = &(_sreturn.newJobs[0]);

	if(Worker_impl::_jobLimit <= 0) {
		// presolver nur, wenn timelimit aufgehoben ist
		_parm2.ps_heur = GLP_ON;
		_parm2.presolve = GLP_ON;
		_parm2.binarize = GLP_ON;
	}

	int statusMIP = glp_intopt(Worker_impl::_lp, &_parm2);
	
	if (Worker_impl::_jobLimit == Worker_impl::TIMEOUT_REACHED) {
		_sreturn.newJobs[0]._bestBound = Worker_impl::_bestBound;
		
		_sreturn.newJobs = JobTool::branch(
			sim,
			_sreturn.newJobs[0],
			&(Worker_impl::_data),
			&(Worker_impl::_conf)
		);
		_sreturn.flag = SolveFlag::TIMEOUT;
		return;
	}

	switch(statusMIP) {
		case GLP_EFAIL:
			/// @todo das muss den Farmer eigentlich abbrechen lassen!!?
			_sreturn.flag = SolveFlag::FAIL;
			return;

		case GLP_ENODFS:
			/// @todo das muss den Farmer eigentlich abbrechen lassen!!?
			_sreturn.flag = SolveFlag::FAIL;
			return;

		case GLP_ENOPFS:
			_sreturn.flag = SolveFlag::NOSOL;
			return;

		case GLP_EBOUND:
			_sreturn.flag = SolveFlag::BOUNDERR;
			return;
	}

	switch(glp_mip_status(Worker_impl::_lp)) {
		case GLP_NOFEAS:
			_sreturn.flag = SolveFlag::NOSOL;
			return;
		
		case GLP_UNDEF:
			/// @todo das muss den Farmer eigentlich abbrechen lassen!!?
			_sreturn.flag = SolveFlag::FAIL;
			return;
	}

	_sreturn.newJobs[0]._fix.resize(n);
	for(int i=0; i<n; ++i) {
		_sreturn.newJobs[0]._fix[i] = glp_mip_col_val(Worker_impl::_lp, i+1);
	}
	_sreturn.newJobs[0]._opt = glp_mip_obj_val(Worker_impl::_lp);
	// SOLVE END -------------------------------------------------------

	_sreturn.flag = SolveFlag::OK;
	Worker_impl::_mu.lock();

	if (
		(JobTool::compare(Worker_impl::_bestJob, _sreturn.newJobs[0]) < 0 &&  Worker_impl::_data.isMax) ||
		(JobTool::compare(Worker_impl::_bestJob, _sreturn.newJobs[0]) > 0 && !Worker_impl::_data.isMax) 
	) {
		Worker_impl::_bestJob = _sreturn.newJobs[0];
	}
	Worker_impl::_mu.unlock();
}


// INPUT ===============================================================
void Worker_impl::input(const Obj & data, const Configurator & conf) {
	if (data.isMax == true) {
		JobTool::BAD_OptNBound = -1.0 * std::numeric_limits<double>::max();
	} else {
		JobTool::BAD_OptNBound = std::numeric_limits<double>::max();
	}
	Worker_impl::_mu.lock();
	Worker_impl::_bestJob = JobTool::create(-1);
	Worker_impl::_mu.unlock();
	Worker_impl::_data = data;
	Worker_impl::_bestBound = JobTool::BAD_OptNBound;
	Worker_impl::_conf = conf;

	glp_init_smcp(&_parm1);
	glp_mem_limit(Worker_impl::_megaMemLimit);
	glp_term_out(GLP_OFF); // we are missing error :-S but heuristic is NOT talking

	if (Worker_impl::_lp != nullptr) glp_delete_prob(Worker_impl::_lp);
	Worker_impl::_lp = glp_create_prob();
	
	if (data.isMax == true) {
		glp_set_obj_dir(Worker_impl::_lp, GLP_MAX);
	} else {
		glp_set_obj_dir(Worker_impl::_lp, GLP_MIN);
	}
	glp_add_rows(Worker_impl::_lp, data.rows.size());
	glp_add_cols(Worker_impl::_lp, data.cols.size());
	
	for (auto col : data.cols) {
		glp_set_col_kind(
			Worker_impl::_lp, 
			col.at,
			col.kind
		);
		glp_set_col_bnds(
			Worker_impl::_lp,
			col.at,
			col.type,
			col.lower,
			col.upper
		);
	}

	int rnum;
	for (auto row : data.rows) {
		glp_set_row_bnds(
			Worker_impl::_lp,
			row.at,
			row.type,
			row.lower,
			row.upper
		);
		
		/// @issue  int[] = vector<int> ??!?!?
		rnum = row.ids.size();
		int rowIds[rnum]{0};
		double rowVals[rnum]{0.0};
		for (int i=0; i <rnum; ++i) {
			rowIds[i+1]  = row.ids[i];
			rowVals[i+1] = row.vals[i];
		}
		
		glp_set_mat_row(
			Worker_impl::_lp,
			row.at,
			rnum,
			rowIds,
			rowVals
		);
	}

	// Bounds + Objectiv
	
	glp_add_rows(Worker_impl::_lp, 1);
	int    idxArr[data.coefs.size() +1];
	double valArr[data.coefs.size() +1];
	idxArr[0] = 0;
	valArr[0] = 0;
	
	for (unsigned i=0; i < data.coefs.size(); ++i) {
		glp_set_obj_coef(Worker_impl::_lp, i+1, data.coefs[i]);
		idxArr[i+1] = i+1;
		valArr[i+1] = data.coefs[i];
	}
	
	glp_set_mat_row(Worker_impl::_lp, data.rows.size()+1, data.coefs.size(), idxArr, valArr);
}


// =====================================================================
void Worker_impl::kill() {
	Worker_impl::_kill = true;
	usleep(500000); // good choice to catch all glpk ?
	Worker_impl::_kill = false;
}


// =====================================================================
void Worker_impl::update(UpdateResult & _return, const Job & bestJob) {
	Worker_impl::_mu.lock();
	if (Worker_impl::_data.isMax == true) {
		if (JobTool::compare(Worker_impl::_bestJob, bestJob) < 0) {
			Worker_impl::_bestJob = bestJob;
		}
	} else {
		if (JobTool::compare(Worker_impl::_bestJob, bestJob) > 0) {
			Worker_impl::_bestJob = bestJob;
		}
	}
	_return.job = Worker_impl::_bestJob;
	Worker_impl::_mu.unlock();
	
	static size_t memory, tpeak;
	static int count, cpeak;
	glp_mem_usage(&count, &cpeak, &memory, &tpeak);

	_return.mem = (memory/10000) / Worker_impl::_megaMemLimit;
}

// =====================================================================
Worker_impl::Worker_impl(const int & megaMemLimit) {
	Worker_impl::_megaMemLimit = megaMemLimit;
}
